 /******************************************************************************
  *       __   _   _____   _   _        _____   _____   _____                  *
  *      |  \ | | | ____| | | | |      /  ___| /  ___| /  ___/                 *
  *      |   \| | | |__   | | | |      | |     | |     | |___                  *
  *      | |\   | |  __|  | | | |      | |     | |     \___  \                 *
  *      | | \  | | |___  | |_| |      | |___  | |___   ___| |                 *
  *      |_|  \_| |_____| \_____/      \_____| \_____| /_____/                 *
  *                                                                            *
  *                          BEACON Example                                    *
  *                           Transmitter                                      *
  ******************************************************************************
  *                                                                            *
  * A MSP430F1611 running this program will broadcast beacon packets in        *
  * different power levels. The broadcasting rate is 4 packets/second.         *
  * - PAN ID: 0xFFFF                                                           *
  * - Short local address: 0x0001                                              *
  * - Destination address: 0xFFFF                                              *
  *                                                                            *
  *                                                                            *
  * INSTRUCTIONS:                                                              *
	*	The payload of a beacon frame consists of 4 bytes.                         *
	* 16 bit sequence number is stored in first two bytes.                       *
	* 8 bit object identification nubmer is stored in the 3rd byte.              *
	* The last 8 bit of TXCTRL register(16bit) which is the configuration reg.   *
	* for output power is stored in the 4th byte.                                *
	*                                                                            *
	* Packet Format:                                                              *
	*	-----------------------------------------------------------------          *
  * |SEQ high 8 bits|SEQ low 8 bits |   Object ID   | TX power level|          *
  * -----------------------------------------------------------------          *
  *                                                                            *
  * Note:                                                                      *
  * For simplification, only three different output power levels will be used. *
  * TXCTRL   Power Level(dbm)                                                  *
  * 0xA0FF   0                                                                 *
  * 0xA0EB   -10                                                               *
  * 0xA0E3   -25                                                               *
  ******************************************************************************
  * Compiler: msp430-gcc GNU C compiler                                        *
  * Target platform: MSP-430F1611 (can easily be ported to other platforms)    *
  ******************************************************************************
  * Last update: 03/19/2006                                                    *
  ******************************************************************************/

#include "include/include.h"
#include "include/hardware.h"
#include "include/swuart.h"
#include <stdlib.h>
#include <stdio.h>

//Defintions used locally in this file.
#define PAYLOAD_SIZE	4
#define RF_CHANNEL	26
#define FILL_BYTE    0xEE

//Basic RF transmission structures.
BASIC_RF_TX_INFO rfTxInfo;
BASIC_RF_RX_INFO rfRxInfo;
BYTE pTxBuffer[BASIC_RF_MAX_PAYLOAD_SIZE];
BYTE pRxBuffer[BASIC_RF_MAX_PAYLOAD_SIZE];
WORD PWLVS[]={0xA0FF,0xA0EB,0xA0E3};//The TX power levels for 0, -10, -25 dbm
INT8 OBJ_ID=4; //OBJ id
WORD ipw=0; //The index for TX power level.
WORD seq=0; //Beacon seq number.


/*
 *  void sendBeacon(void)
 *
 */
void sendBeacon(void){
	seq++;
	WORD outpw=PWLVS[(ipw++)%3];
	FASTSPI_SETREG(CC2420_TXCTRL, outpw);
	pTxBuffer[0]= (seq>>8)&0xFF;
	pTxBuffer[1]= seq&0xFF;
	pTxBuffer[2]= OBJ_ID;
	pTxBuffer[3]= outpw&0xFF;
	//printf("Sending a packet.Seq=%u, Obj Id=%u, Power Level:0x%X\r\n",(pTxBuffer[0]<<8)+pTxBuffer[1],pTxBuffer[2],pTxBuffer[3]);
	TOGGLE_BLED();
	basicRfSendPacket(&rfTxInfo);

} //run

/*
 *  int main (void)
 *
 *  DESCRIPTION:
 *    Startup routine and main loop
 */
int main (void) {
	//Initalize ports for communication with CC2420 and other peripheral units.
	PORT_INIT();
	SPI_INIT();
	SWUART_INIT();

	//Initialize for basic RF operation.
	basicRfInit(&rfRxInfo, RF_CHANNEL, 0xFFFF, 0x0001);
	rfTxInfo.destAddr = 0xFFFF;
	
	//Initialize common protocol parameters.
	rfTxInfo.length = PAYLOAD_SIZE;
	rfTxInfo.ackRequest = FALSE;
	rfTxInfo.pPayload = pTxBuffer;
  
	//Initialize the payload buffer to the default value.
	UINT8 n;
	for (n = 0; n < PAYLOAD_SIZE; n++)
		pTxBuffer[n] = FILL_BYTE;

	//In order to get the ACK, set RF receive to on also.
	//basicRfReceiveOn();	


	//Timer B initialization for beacon using ACLK, up mode.
	TBCCTL0 = CCIE;                    
  TBCCR0 = 2048;
  TBCTL = TBSSEL_1 + MC_1;      
	
	//The main loop:
	while (1){
		  _BIS_SR(LPM0_bits + GIE);       
			
	}
} //main


/*
 *  BASIC_RF_RX_INFO* basicRfProcessPacket(BASIC_RF_RX_INFO *pRRI)
 *
 *  DESCRIPTION:
 *    This function is a part of the basic RF library, but must be declared by 
 *    the application. Once the application has turned on the receiver, using 
 *    basicRfReceiveOn(), all incoming packets will be received by the FIFOP 
 *    interrupt service routine. When finished, the ISR will call the 
 *    basicRfProcessPacket() function. Please note that this function must 
 *    return quickly, since the next received packet will overwrite the active
 *    BASIC_RF_RX_INFO structure (pointed to by pRRI).
 *  Notes:
 *    Although the current program is only for the transmitting, the RF receive
 *    capability is still necessary if ACK is used in the transmission. But if 
 *    the packet received is an ACK, this function won't be invoked.
 *  ARGUMENTS:
 *    BASIC_RF_RX_INFO *pRRI
 *      The reception structure, which contains all relevant info about the 
 *      received packet.
 *
 *  RETURN VALUE:
 *    BASIC_RF_RX_INFO*
 *      The pointer to the next BASIC_RF_RX_INFO structure to be used by the 
 *      FIFOP ISR. If there is only one buffer, then return pRRI.
 */
BASIC_RF_RX_INFO* basicRfProcessPacket(BASIC_RF_RX_INFO *pRRI) {
    //Process the received data.
    return pRRI;
} //basicRfReceivePacket


/*
 *  CC2420 FIFOP interrupt service routine (ISR)
 *
 *  DESCRIPTION:
 *    When a packet has been completely received, this ISR will extract the data
 *    from the RX FIFO, put it into the active BASIC_RF_RX_INFO structure, and 
 *    call basicRfProcessPacket() (defined by the application). FIFO overflow 
 *    and illegally formatted packets are handled by this routine.
 *
 *  Note1: Packets are acknowledged automatically by CC2420 through the auto- 
 *    acknowledgment feature.
 *  Note2: ACK will also be received in thie ISR but basicRfProcessPacket() 
 *    won't get invoked.
 */
//interrupt (PORT1_VECTOR) fifo_rx(void){
//    //Clearing FIFOP Interrupt.
//    CLEAR_FIFOP_INT();
//		
//    basicRfReceivePacket();
//} //fifo_rx


interrupt (TIMERB0_VECTOR) Timer_B (void)
{
		DISABLE_GLOBAL_INT();
  	sendBeacon();
  	ENABLE_GLOBAL_INT();
}

